// Nom du snack : eni_react_native_villes_modification
// Nom du script : App.js
// Auteur : Christian VIGOUROUX
// Date de création : 11/07/2024
// Date de dernière modification : 11/07/2024 (Christian VIGOUROUX)
// Objet : Affichage d'une liste de villes avec une lecture depuis une table MySQL et possibilité de suppression
// URL : https://snack.expo.dev/@cvigouroux/eni_react_native_villes_modification

// Import des composants React Native et React
import React, { useState, useEffect } from 'react';
import { SafeAreaView, Text, StyleSheet, View, FlatList, Button, Alert, Modal, TextInput, TouchableOpacity } from 'react-native';

// Fonction principale App
export default function App() {
  
  // Constantes de la fonction App
  const [dataSource, setDataSource] = useState([]);
  const [modalVisible, setModalVisible] = useState(false);
  const [selectedCity, setSelectedCity] = useState({});
  const [newCityName, setNewCityName] = useState('');

  // Méthode useEffect récupérant les données
  // NB : Accès aux données de l'application distante (format JSON)
  //      URL : http://christian-vigouroux.fr/react/php_mysql_liste_complete_villes.php
  useEffect(() => {
    fetch('http://christian-vigouroux.fr/react/php_mysql_liste_complete_villes.php')
      .then((response) => response.json())
      .then((responseJson) => {
        setDataSource(responseJson);
      })
      .catch((error) => {
        console.error(error);
        Alert.alert('Erreur', 'Impossible de récupérer les données');
      });
  }, []);

  // Fonction de déclenchement de la modification
  // NB : Ouverture du modal et initialisation des valeurs de modification
  const handleModify = (item) => {
    setSelectedCity(item);
    setNewCityName(item.nom_ville);
    setModalVisible(true);
  };

  // Fonction d'enregistrement de la modification
  // NB : Envoi de la nouvelle valeur à l'API et mise à jour de la liste
  const handleSave = () => {
    fetch('http://christian-vigouroux.fr/react/php_mysql_modification_ville.php', {
      method: 'POST',
      headers: {
        'Accept': 'application/json',
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({ id_ville: selectedCity.id_ville, nom_ville: newCityName })
    })
    .then((response) => {
      console.log('HTTP Status:', response.status);
      console.log('Response Headers:', response.headers);
      if (!response.ok) {
        throw new Error('Network response was not ok');
      }
      return response.json();
    })
    .then((responseJson) => {
      console.log('Response JSON:', responseJson);
      if (responseJson.success) {
        setDataSource(responseJson.data);
        setModalVisible(false);
        Alert.alert('Succès', 'La ville a été mise à jour');
      } else {
        console.error('Update failed: ', responseJson.message);
        Alert.alert('Erreur', responseJson.message);
      }
    })
    .catch((error) => {
      console.error('Fetch Error:', error.message);
      Alert.alert('Erreur', 'Une erreur est survenue : ' + error.message);
    });
  };

  // Fonction ItemView
  // NB : Définition de l'élément de la Flat List
  const ItemView = ({ item }) => {
    return (

      // Vue pour chaque élément de la liste
      <View style={styles.itemContainer}>
        <Text style={styles.itemStyle}>
          {/* Affichage des champs id_ville et nom_ville */}
          {item.id_ville} - {item.nom_ville}
        </Text>
        {/* Affichage du bounton de modification */}
        <Button title="Modification" onPress={() => handleModify(item)} />
      </View>

    );

  };

  // Valeur de retour de la fonction App
  return (
    
    // Vue de type SafeAreaView
    <SafeAreaView style={{ flex: 1 }}>
      
      {/* Vue d'affichage de la Flat List */}
      <View style={styles.container}>
        
        {/* Composant FlatList */}
        <FlatList
          data={dataSource}
          keyExtractor={(item, index) => index.toString()}
          // Affichage de l'item
          renderItem={ItemView}
          // Affichage du trait de séparation
          ItemSeparatorComponent={() => <View style={styles.separator} />}
        />
        
        {/* Composant Modal pour la modification */}
        <Modal
          animationType="slide"
          transparent={true}
          visible={modalVisible}
          onRequestClose={() => {
            setModalVisible(false);
          }}>
          <View style={styles.centeredView}>
            <View style={styles.modalView}>
              <Text>Modification de la ville</Text>
              {/* Champ de texte pour saisir le nouveau nom */}
              <TextInput
                style={styles.input}
                onChangeText={setNewCityName}
                value={newCityName}
              />
              <TouchableOpacity
                style={[styles.button, styles.buttonClose]}
                onPress={handleSave}>
                <Text style={styles.textStyle}>Enregistrement</Text>
              </TouchableOpacity>
            </View>
          </View>
        </Modal>
      
      </View>
    
    </SafeAreaView>
  
  );

}

// Styles CSS
const styles = StyleSheet.create({
  container: {
    backgroundColor: 'white',
    flex: 1,
  },
  itemContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    padding: 10,
  },
  itemStyle: {
    fontSize: 18,
  },
  separator: {
    height: 1,
    backgroundColor: '#C8C8C8',
    width: '100%',
  },
  centeredView: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    marginTop: 22,
  },
  modalView: {
    margin: 20,
    backgroundColor: 'white',
    borderRadius: 20,
    padding: 35,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 2,
    },
    shadowOpacity: 0.25,
    shadowRadius: 4,
    elevation: 5,
  },
  button: {
    borderRadius: 20,
    padding: 10,
    elevation: 2,
  },
  buttonClose: {
    backgroundColor: '#2196F3',
  },
  textStyle: {
    color: 'white',
    fontWeight: 'bold',
    textAlign: 'center',
  },
  input: {
    height: 40,
    margin: 12,
    borderWidth: 1,
    padding: 10,
    width: 200,
  },
});